<?php
/**
 * Custom functions that act independently of the theme templates
 *
 * Eventually, some of the functionality here could be replaced by core features.
 *
 * @package HitMag Pro
 */

/**
 * Adds custom classes to the array of body classes.
 *
 * @param array $classes Classes for the body element.
 * @return array
 */
function hitmag_pro_body_classes( $classes ) {
	global $post;

	if ( false == get_theme_mod( 'hitmag_pro_boxed_layout', true ) ) {
		$classes[] = 'hitmag-full-width';
	}

	// Adds a class of group-blog to blogs with more than 1 published author.
	if ( is_multi_author() ) {
		$classes[] = 'group-blog';
	}

	// Adds a class of hfeed to non-singular pages.
	if ( ! is_singular() ) {
		$classes[] = 'hfeed';
	}

	if ( is_home() || is_archive() || is_search() ) {
		$archive_sidebar_align = get_option( 'archive_sidebar_align', 'th-right-sidebar' );
		$classes[] = esc_attr( $archive_sidebar_align );
	}

	if ( is_single() ) {
		$post_specific_layout = get_post_meta( $post->ID, '_hitmag_pro_layout_meta', true );
		if ( empty( $post_specific_layout ) || $post_specific_layout == 'th-default-layout' ) {
			$classes[] = esc_attr( get_option( 'post_sidebar_align', 'th-right-sidebar' ) );
		} else {
			$classes[] = esc_attr( $post_specific_layout );
		}
	}

	if ( is_page() ) {
		$page_specific_layout = get_post_meta( $post->ID, '_hitmag_pro_layout_meta', true );
		if ( empty( $page_specific_layout ) || $page_specific_layout == 'th-default-layout' ) {
			$classes[] = esc_attr( get_option( 'page_sidebar_align', 'th-right-sidebar' ) );
		} else {
			$classes[] = esc_attr( $page_specific_layout );
		}		
	}

	return $classes;
}
add_filter( 'body_class', 'hitmag_pro_body_classes' );

/**
 * Add a pingback url auto-discovery header for singularly identifiable articles.
 */
function hitmag_pro_pingback_header() {
	if ( is_singular() && pings_open() ) {
		echo '<link rel="pingback" href="', esc_url( get_bloginfo( 'pingback_url' ) ), '">';
	}
}
add_action( 'wp_head', 'hitmag_pro_pingback_header' );

/**
 * Add a custom excerpt length.
 */
function hitmag_pro_excerpt_length( $length ) {
	if( is_admin() ) {
		return $length;
	}
	$custom_length = get_theme_mod( 'excerpt_length', 30 );
	return absint( $custom_length );
}
add_filter( 'excerpt_length', 'hitmag_pro_excerpt_length', 999 );

/**
 * Changes the excerpt more text.
 */
function hitmag_pro_excerpt_more( $more ) {

	if ( is_admin() ) {
		return $more;
	}

	return ' &hellip; ';
}
add_filter( 'excerpt_more', 'hitmag_pro_excerpt_more' );



if ( ! function_exists( 'hitmag_pro_get_layout' ) ) :
/**
 * Returns the selected sidebar alignment for the page.
 *
 * @return string
 */
function hitmag_pro_get_layout() {

	global $post;

	$layout = 'th-right-sidebar';

	if ( is_home() || is_archive() || is_search() ) {
		$archive_sidebar_align = get_option( 'archive_sidebar_align', 'th-right-sidebar' );
		$layout = $archive_sidebar_align;
	}

	if ( is_single() ) {
		$post_specific_layout = get_post_meta( $post->ID, '_hitmag_pro_layout_meta', true );
		if ( empty( $post_specific_layout ) || $post_specific_layout == 'th-default-layout' ) {
			$layout = get_option( 'post_sidebar_align', 'th-right-sidebar' );
		} else {
			$layout = $post_specific_layout;
		}
	}

	if ( is_page() ) {
		$page_specific_layout = get_post_meta( $post->ID, '_hitmag_pro_layout_meta', true );
		if ( empty( $page_specific_layout ) || $page_specific_layout == 'th-default-layout' ) {
			$layout = get_option( 'page_sidebar_align', 'th-right-sidebar' );
		} else {
			$layout = $page_specific_layout;
		}		
	}

	return $layout;

}

endif;


/**
 * Changes tag font size.
 */
function hitmag_pro_tag_cloud_sizes($args) {
	$args['smallest']	= 10;
	$args['largest'] 	= 10;
	return $args; 
}
add_filter('widget_tag_cloud_args','hitmag_pro_tag_cloud_sizes');


/**
 * View all link for posts widgets 
 */
function hitmag_pro_viewall_link( $category_id, $viewall_text ) {

	if ( ! empty( $viewall_text ) ) :

		if ( ! empty( $category_id ) ) {
			$viewall_link = get_category_link( $category_id );
		} else {
			$posts_page_id = get_option( 'page_for_posts' );

			if ( $posts_page_id ) {
				$viewall_link = get_page_link( $posts_page_id );
			} else {
				$viewall_link = "";
			}
		}

		if ( $viewall_link ) { ?>
			<a class="hm-viewall" href="<?php echo esc_url( $viewall_link ); ?>"><span><?php echo esc_html( $viewall_text ); ?></span></a>
		<?php }

	endif;  

}


function hitmag_pro_get_attachment_id( $url ) {
	$attachment_id = 0;
	$dir = wp_upload_dir();
	if ( false !== strpos( $url, $dir['baseurl'] . '/' ) ) { // Is URL in uploads directory?
		$file = basename( $url );
		$query_args = array(
			'post_type'   => 'attachment',
			'post_status' => 'inherit',
			'fields'      => 'ids',
			'meta_query'  => array(
				array(
					'value'   => $file,
					'compare' => 'LIKE',
					'key'     => '_wp_attachment_metadata',
				),
			)
		);
		$query = new WP_Query( $query_args );
		if ( $query->have_posts() ) {
			foreach ( $query->posts as $post_id ) {
				$meta = wp_get_attachment_metadata( $post_id );
				$original_file       = basename( $meta['file'] );
				$cropped_image_files = wp_list_pluck( $meta['sizes'], 'file' );
				if ( $original_file === $file || in_array( $file, $cropped_image_files ) ) {
					$attachment_id = $post_id;
					break;
				}
			}
		}
	}
	return $attachment_id;
}

function hitmag_pro_single_post_template() {

	global $post;

	if ( ! is_single() ) {
		return;
	}

	$post_specific_template = get_post_meta( $post->ID, '_hitmag_pro_post_layout', true );

	if ( empty( $post_specific_template ) || $post_specific_template == 'post-style-default' ) {
		$layout = get_option( 'hitmag_pro_post_template', 'post-style-1' );
		return $layout;
	} else {
		return $layout = $post_specific_template;
	}

}