<?php

/**
 * Displays latest, category wised posts in a 3 block layout.
 */

class HitMag_Pro_Grid_Category_Posts extends WP_Widget {

	/* Register Widget with WordPress*/
	function __construct() {
		parent::__construct(
			'hitmag_pro_grid_category_posts', // Base ID
			__( 'Magazine Posts ( Style 3 )', 'hitmag-pro' ), // Name
			array( 'description' => __( 'Displays latest posts or posts from a choosen category.', 'hitmag-pro' ), ) // Args
		);
	}


	/**
	 * Back-end widget form.
	 *
	 * @see WP_Widget::form()
	 *
	 * @param array $instance Previously saved values from database.
	 */

	public function form( $instance ) {
		//print_r($instance);
		$defaults = array(
			'title'			=> __( 'Latest Posts', 'hitmag-pro' ),
			'category'		=> 'all',
			'number_posts'	=> 6,
			'sticky_posts' 	=> true,
			'viewall_text'	=> __( 'View All', 'hitmag-pro' )
		);
		$instance = wp_parse_args( (array) $instance, $defaults );

	?>

		<p>
			<label for="<?php echo $this->get_field_id( 'title' ); ?>"><?php _e( 'Title:', 'hitmag-pro' ); ?></label>
			<input type="text" class="widefat" id="<?php echo $this->get_field_id( 'title' ); ?>" name="<?php echo $this->get_field_name( 'title' ); ?>" value="<?php echo esc_attr($instance['title']); ?>"/>
		</p>
		<p>
			<label><?php _e( 'Select a post category:', 'hitmag-pro' ); ?></label>
			<?php wp_dropdown_categories( array( 'name' => $this->get_field_name('category'), 'selected' => $instance['category'], 'show_option_all' => 'Show all posts', 'class' => 'widefat' ) ); ?>
		</p>
		<p>
			<label for="<?php echo $this->get_field_id( 'number_posts' ); ?>"><?php _e( 'Number of posts:', 'hitmag-pro' ); ?></label>
			<input class="widefat" type="number" id="<?php echo $this->get_field_id( 'number_posts' ); ?>" name="<?php echo $this->get_field_name( 'number_posts' );?>" value="<?php echo absint( $instance['number_posts'] ); ?>" size="3"/> 
		</p>
		<p>
			<input type="checkbox" <?php checked( $instance['sticky_posts'], true ) ?> class="checkbox" id="<?php echo $this->get_field_id('sticky_posts'); ?>" name="<?php echo $this->get_field_name('sticky_posts'); ?>" />
			<label for="<?php echo $this->get_field_id('sticky_posts'); ?>"><?php _e( 'Hide sticky posts.', 'hitmag-pro' ); ?></label>
		</p>
		<p>
			<label for="<?php echo $this->get_field_id( 'viewall_text' ); ?>"><?php _e( 'View All Text:', 'hitmag-pro' ); ?></label>
			<input type="text" class="widefat" id="<?php echo $this->get_field_id( 'viewall_text' ); ?>" name="<?php echo $this->get_field_name( 'viewall_text' ); ?>" value="<?php echo esc_attr( $instance['viewall_text'] ); ?>"/>
		</p>			
	<?php

	}



	/**
	 * Sanitize widget form values as they are saved.
	 *
	 * @see WP_Widget::update()
	 *
	 * @param array $new_instance Values just sent to be saved.
	 * @param array $old_instance Previously saved values from database.
	 *
	 * @return array Updated safe values to be saved.
	 */
	
	public function update( $new_instance, $old_instance ) {
		$instance = $old_instance;
		$instance[ 'title' ] = sanitize_text_field( $new_instance[ 'title' ] );	
		$instance[ 'category' ]	= absint( $new_instance[ 'category' ] );
		$instance[ 'number_posts' ] = (int)$new_instance[ 'number_posts' ];
		$instance[ 'sticky_posts' ] = (bool)$new_instance[ 'sticky_posts' ];
		$instance[ 'viewall_text' ] = sanitize_text_field( $new_instance[ 'viewall_text' ] );
		return $instance;
	}


	/**
	 * Front-end display of widget.
	 *
	 * @see WP_Widget::widget()
	 *
	 * @param array $args     Widget arguments.
	 * @param array $instance Saved values from database.
	 */
	
	public function widget( $args, $instance ) {
		extract($args);

		$title = ( ! empty( $instance['title'] ) ) ? $instance['title'] : '';	
        $title = apply_filters( 'widget_title', $title , $instance, $this->id_base );
		$category = ( ! empty( $instance['category'] ) ) ? absint( $instance['category'] ) : '';
		$number_posts = ( ! empty( $instance['number_posts'] ) ) ? absint( $instance['number_posts'] ) : 6; 
		$sticky_posts = ( isset( $instance['sticky_posts'] ) ) ? $instance['sticky_posts'] : true;
		$viewall_text = ( ! empty( $instance['viewall_text'] ) ) ? $instance['viewall_text'] : '';
		// Latest Posts
		$latest_posts = new WP_Query( 
			array(
				'cat'					=> $category,
				'posts_per_page'		=> $number_posts,
				'post_status'			=> 'publish',
				'ignore_sticky_posts' 	=> $sticky_posts,
				)
		);	

		$hm_current_page = 1;

		if( $latest_posts->max_num_pages <= $hm_current_page) {
			$show_next = false;
		} else {
			$show_next = true;
		}		

		$ajax_parameters = array(
			'hm_block_uid' 		=> $widget_id,
			'hm_category'		=> $category,
			'hm_number_posts'	=> $number_posts,
			'hm_current_page'	=> $hm_current_page,
			'hm_posts_per_page' => $number_posts,
			'hm_show_next'		=> $show_next,
			'hm_show_prev'		=> false
		);		

		echo $before_widget;

		if ( ! empty($title) ) {
			echo $before_title. $title . $after_title;
		}

		hitmag_pro_viewall_link( $category, $viewall_text );

		echo '<div class="hmw-spinner"><div class="rect1"></div><div class="rect2"></div><div class="rect3"></div><div class="rect4"></div><div class="rect5"></div></div>';
		echo '<div class="hitmag-block">';

			self::render_view( $latest_posts, $ajax_parameters, false );

		echo '</div>';

		echo $after_widget;

	}

	public static function render_view( $latest_posts, $ajax_parameters, $hm_is_ajax = false ) { ?>

		<div class="hitmag-grid-category-posts">

            <?php 

                if ( $latest_posts -> have_posts() ) :

                while ( $latest_posts -> have_posts() ) : $latest_posts -> the_post(); ?>

                    <div class="hmw-grid-post">
                        <div class="hm-grid-thumb">
                            <?php if ( has_post_thumbnail() ) { ?>
                                <a href="<?php the_permalink(); ?>" title="<?php the_title_attribute(); ?>">
	                            	<?php 
	                            		the_post_thumbnail( 'hitmag-grid' ); 

										if ( has_post_format( 'video' ) ) {
											hitmag_pro_show_video_icon( 'medium' );
										} 
									?>
                                </a>
                            <?php } ?>
                        </div>
                        <div class="hm-grid-details">
                            <?php the_title( sprintf( '<h2 class="post-title"><a href="%s" rel="bookmark">', esc_url( get_permalink() ) ), '</a></h2>' ); ?>
                            <p class="hms-meta"><?php echo hitmag_pro_posted_datetime(); ?></p>
                        </div>
                    </div>

            <?php 
                endwhile; 
                wp_reset_postdata();
            endif; ?>

		</div><!-- .hitmag-grid-category-posts -->	

		<?php hitmag_pro_ajax_pagination( $ajax_parameters );	
	}


}

// Register single category posts widget
function hitmag_pro_register_grid_category_posts() {
    register_widget( 'HitMag_Pro_Grid_Category_Posts' );
}
add_action( 'widgets_init', 'hitmag_pro_register_grid_category_posts' );